class HubController : Thinker {
    
    static void setupHub(void)
    {
        RampDataLibrary.inst().refreshDataArrays();
        DNLogger.Log("INIT", "Refreshed data arrays");
        
        int skill = G_SkillPropertyInt(SKILLP_ACSReturn);
        RampDataLibrary.WriteInt("CurrentSkill", skill);
        
        PlayerPawn p = PlayerPawn(players[consoleplayer].mo);
        p.player.cheats = 0;
        //If we're coming back from a level, set the completion status accordingly
        int mapCameFrom = RampDataLibrary.GetCurrentMap(null);
        if (mapCameFrom > 0) {
            //console.printf("REWARDDEBUG: Came from map " .. mapCameFrom);
            //If this is newly completed, award the level's cash value to the player
            if (RampDataLibrary.ReadInt("MapCompleted" .. mapCameFrom) == 0)
            {
                int levelCash = RampDataLibrary.ReadInt("MapCash" .. mapCameFrom);
                //console.printf("REWARDDEBUG: Map cash was " .. levelCash);
                levelCash = (levelCash * RampDataLibrary.inst().LEVEL_VALUE_PERCENT)/100;
                //RampDataLibrary.givePlayerCash(levelCash);
                RampDataLibrary.WriteInt("LevelReturnCash", levelCash);
            }
            RampDataLibrary.WriteInt("MapCompleted" .. mapCameFrom, 1);
            
            //Compare the number of secrets we had before entering to the number of secrets found now - award per found secret
            int secretsNewlyFound = RampDataLibrary.ReadInt("MapFoundSecrets" .. mapCameFrom) - RampDataLibrary.ReadInt("MapSecretsOnLeavingHub");
            if (secretsNewlyFound) {
                //console.printf("REWARDDEBUG: Map secrets newly found: " .. secretsNewlyFound);
                int secretsCash = secretsNewlyFound * RampDataLibrary.inst().SECRET_VALUE;
                //RampDataLibrary.givePlayerCash(secretsCash);
                RampDataLibrary.WriteInt("LevelReturnSecretsCash", secretsCash);
                RampDataLibrary.WriteInt("LevelReturnSecrets", secretsNewlyFound);                
            }
            RampDataLibrary.WriteInt("MapSecretsOnLeavingHub", 0);
        }

        RampDataLibrary.inst().typedMapNumber = "";
        setupPortals();
        RampDataLibrary.inst().UpdateStationCounts();
        if (RampDataLibrary.ReadInt("GameSetupDone") == 0) {
            RampDataLibrary.inst().createMissionArray();
            RampDataLibrary.WriteInt("GameSetupDone", 1);
            DNLogger.Log("INIT", "Performed initial game setup");
        }
        setupSubwayScreens();
        UpdateMissionDisplay();
        DrawEndingScreen();
    }
    
    static void setupSubwayScreens(void) {  
        int lineTag = 2200;
        for (int i = 0; i < RampDataLibrary.inst().missionArray.size(); i++) {
            RampMissionInfo info = RampDataLibrary.inst().missionArray[i];
            //DNLogger.Log("INIT", "Station entry: " .. info.stationTag .. " " .. info.stationY);
            Line myLine = RampDataLibrary.GetLineWithTag(lineTag);
            if (myLine) {
                
                int myYoffset = 0 - 110 - info.stationY;
                
                Side s = myLine.sidedef[1];
                
                // Only display the station name if this station is unlocked
                if (i >= RampDataLibrary.inst().stationsUnlocked) {
                    s.SetTexture(2, TexMan.CheckForTexture("RSTN2000", TexMan.Type_MiscPatch));
                }
                else {
                    s.SetTexture(2, TexMan.CheckForTexture("RSTN" .. info.stationTag, TexMan.Type_MiscPatch));
                }
                s.SetTextureYOffset(2, myYoffset);
                s.setTextureXOffset(2, 0);
                
                //If this isn't the last one, get the connecting line as well and work out its texture and offset
                if (i < RampDataLibrary.inst().missionArray.size()-1) {
                    Line connectingLine = RampDataLibrary.GetLineWithTag(lineTag + 101);
                    if (connectingLine) {
                        s = connectingLine.sidedef[1];
                        // Set the initial Y offset to match the station...
                        s.SetTextureYOffset(2, myYoffset);
                        s.setTextureXOffset(2, 0);
                        // Decide the texture based on the difference between this and the next one
                        RampMissionInfo nextInfo = RampDataLibrary.inst().missionArray[i+1];
                        int difference = info.stationY - nextInfo.stationY;
                        int steps = abs(difference) / 80;
                        string dir = "DN";
                        if (difference > 0) {
                            dir = "UP";
                        }
                        else if (difference == 0) {
                            dir = "ST";
                        }
                        string connectorString = dir .. steps .. Random(1, 2);
                        s.SetTexture(2, TexMan.CheckForTexture("RSTC" .. connectorString, TexMan.Type_MiscPatch));
                    }
                }
                
                //If this is the first one in a carriage, do the same to the previous line (terminus)
                if (i % 6 == 0) {
                    Line startConnectingLine = RampDataLibrary.GetLineWithTag(lineTag + 100);
                    if (startConnectingLine) {
                        s = startConnectingLine.sidedef[1];
                        s.SetTextureYOffset(2, myYoffset);
                        s.setTextureXOffset(2, 0);
                        string startConnectorString = i == 0 ? "TMR2" : "TMR1";
                        s.SetTexture(2, TexMan.CheckForTexture("RSTC" .. startConnectorString, TexMan.Type_MiscPatch));
                    }
                }
                
                //If this is at the end of a carriage, set the next line to the same height as well
                if ((i+1)%6 == 0 || i == 38) {
                    DNLogger.log("INIT", "" .. lineTag + 200);
                    Line afterLine = RampDataLibrary.GetLineWithTag(lineTag + 200);
                    if (afterLine) {
                        s = afterLine.sidedef[1];
                        s.SetTextureYOffset(2, myYoffset);
                        s.setTextureXOffset(2, 0);
                        
                        //Use the continuing line unless this is the end (check game mode)
                        string endConnectorString = "TML1";
                        int stationsInGame = min(RampDataLibrary.ReadInt("GameMode") * 6, 39);
                        DNLogger.log("INIT", "Stations in game: " .. stationsInGame .. ", drawing " .. i);
                        if (stationsInGame > 0 && i == stationsInGame - 1) {
                            endConnectorString = "TML2";
                        }
                        s.SetTexture(2, TexMan.CheckForTexture("RSTC" .. endConnectorString, TexMan.Type_MiscPatch));
                    }
                }
            }
            lineTag++;
        }
    }

    static void setupPortals(void)
    {
        RampDataLibrary.inst().mapsCompleted = 0;

        RampPortal spot; ThinkerIterator it = ThinkerIterator.Create("RampPortal");
        //Retrieve an appropriate level according to the zone
        while (spot = RampPortal(it.Next()))
        {
            int tag = spot.tid;
            int portalZone = spot.args[0];
            RampMapInfo levelInfo = RampDataLibrary.GetNextMapInfoForZone(portalZone);            
            if (levelInfo == null) {
                DNLogger.Log("PORTALS", String.Format("Ran out of available levels for zone " .. portalZone .. ", some will be blank"));
                continue;
            }
            //This map info is from when the map was originally loaded, so refresh it from the master list
            levelInfo = RampDataLibrary.Maps(levelInfo.levelnum);            
            DNLogger.Log("PORTALS", "Placing map from master list: " .. levelInfo.toString());
            bool success = setupPortal(spot, levelInfo);
            if (!success) {
                DNLogger.Log("PORTALS", String.Format("Map with slot number %d didn't exist - explicit map portal disabled", tag));
                continue;
            }
        }
    }
    
    static bool setupPortal(RampPortal spot, RampMapInfo mapInfo) {
        int mapNumber = mapInfo.levelnum;
        //DNLogger.Log("PORTALS", String.Format("Setting up map %d", mapNumber));
        
        String mapNameLangKey = mapInfo.lumpName .. "NAME";
        String mapname = StringTable.Localize(mapNameLangKey);
        int mapCompleted = RampDataLibrary.ReadInt("MapCompleted" .. mapNumber);

        if (mapNameLangKey == ("$" .. mapname))
        {
            //This map is not present
            spot.curSector.SetTexture(0, TexMan.CheckForTexture("RGATEGRA", TexMan.Type_MiscPatch));
            return false;
        }

        //Spawn light around the point - these will be coloured blue or yellow depending on completion status
        Array<PointLightAttenuated> spawnedLights;
        spawnedLights.Push(PointLightAttenuated(Actor.Spawn("PointLightAttenuated", (spot.pos.x, spot.pos.y, spot.pos.z + 16))));

        //Specific colours for length/difficulty
        vector2 marqueeLightVec;
        marqueeLightVec = spot.Vec2Angle(93, spot.angle + 95);
        PointLightAttenuated diffLight = PointLightAttenuated(Actor.Spawn("PointLightAttenuated", (marqueeLightVec.x, marqueeLightVec.y, spot.pos.z + 80)));
        marqueeLightVec = spot.Vec2Angle(93, spot.angle - 95);
        PointLightAttenuated lengthLight = PointLightAttenuated(Actor.Spawn("PointLightAttenuated", (marqueeLightVec.x, marqueeLightVec.y, spot.pos.z + 80)));
        if (diffLight != null) { diffLight.A_SetSpecial(0, 255, 64, 64, 32); }
        if (lengthLight != null) { lengthLight.A_SetSpecial(0, 180, 180, 255, 32); }

        
        vector3 lightprops[9] = {
            (120, 160, 64),
            (-120, 160, 64),
            (80, 121, -16),
            (-80, 121, -16),
            (55, 131, -16),
            (-55, 131, -16),
            (30, 101, -16),
            (-30, 101, -16),
            (0, 94, -16)
        };
        int sizes[10] = {80, 64, 64, 32, 32, 32, 32, 32, 32, 32};
        
        for (int i = 0; i < lightprops.Size(); i++) {
            marqueeLightVec = spot.Vec2Angle(lightprops[i].y, spot.angle + lightprops[i].x);
            spawnedLights.Push(PointLightAttenuated(Actor.Spawn("PointLightAttenuated", (marqueeLightVec.X, marqueeLightVec.Y, spot.pos.z + lightprops[i].z))));
        }
        
        if (mapCompleted == 1) {
            RampDataLibrary.inst().mapsCompleted += 1;
            int zoneCount = RampDataLibrary.ReadInt("MapsPerZoneCompleted" .. mapInfo.zone);
            RampDataLibrary.WriteInt("MapsPerZoneCompleted" .. mapInfo.zone, ++zoneCount);
            
            

            for (int i = 0; i < spawnedLights.Size(); i++) {
                spawnedLights[i].A_SetSpecial(0, 64, 64, 255, sizes[i]); // Blue
            }
            Actor.Spawn("RampLevelMarkerA", (spot.pos.x, spot.pos.y, spot.pos.z + 64));
            Actor.Spawn("RampLevelHoverA", (spot.pos.x, spot.pos.y, spot.pos.z + 64));
        }
        else {
            for (int i = 0; i < spawnedLights.Size(); i++) {
                spawnedLights[i].A_SetSpecial(0, 255, 255, 64, sizes[i]); // Yellow
            }
            Actor.Spawn("RampLevelMarkerD", (spot.pos.x, spot.pos.y, spot.pos.z + 64));
            Actor.Spawn("RampLevelHoverD", (spot.pos.x, spot.pos.y, spot.pos.z + 64));
        }
        
        // Spawn enter/exit sector actors
        vector2 enterExitVec = spot.Vec2Angle(60, spot.angle);
        let borderMapSpot = Actor.Spawn("MapSpot", (enterExitVec.X, enterExitVec.Y, spot.pos.z + 4));
        Sector borderSector = borderMapSpot.curSector;
        
        enterExitVec = spot.Vec2Angle(52, spot.angle);
        let skirtMapSpot = Actor.Spawn("MapSpot", (enterExitVec.X, enterExitVec.Y, spot.pos.z + 4));
        Sector skirtSector = skirtMapSpot.curSector;
        
        enterExitVec = spot.Vec2Angle(72, spot.angle - 180);
        let backMapSpot = Actor.Spawn("MapSpot", (enterExitVec.X, enterExitVec.Y, spot.pos.z + 4));
        Sector backSector = backMapSpot.curSector;
        
        enterExitVec = spot.Vec2Angle(88, spot.angle);
        let spawnedSectorExit = Actor.Spawn("MapSpot", (enterExitVec.X, enterExitVec.Y, spot.pos.z));
        
        
        let spawnedLevelEnter = Actor.Spawn("RampGateEnter", (spot.pos.x, spot.pos.y, spot.pos.z + 64));
        spawnedLevelEnter.A_SetSpecial(80, 997, 0, mapNumber);
        
        int mapCash = RampDataLibrary.ReadInt("MapCash" .. mapNumber);
        
        //Position the player here as well if they came from this level
        if (RampDataLibrary.GetCurrentMap(null) == mapNumber) {
            PlayerPawn p = PlayerPawn(players[consoleplayer].mo);
            double sectorFloorHeight;
            Sector sector;
            [sectorFloorHeight, sector] = spawnedSectorExit.curSector.LowestFloorAt((spawnedSectorExit.pos.X, spawnedSectorExit.pos.Y));
            p.SetOrigin((spawnedSectorExit.pos.X, spawnedSectorExit.pos.Y, sectorFloorHeight), false);
            p.angle = spot.angle;
        }
        
        // Spawn traffic cones for WIPs!
        if (mapInfo.isWip == 1)
        {
            vector2 coneOffset = spot.Vec2Angle(70, spot.angle + 60);
            vector2 coneOffset2 = spot.Vec2Angle(70, spot.angle - 60);
            let spawnedCone = Actor.Spawn("TrafficCone", (coneOffset.X, coneOffset.Y, spot.pos.z+16));
            let spawnedCone2 = Actor.Spawn("TrafficCone", (coneOffset2.X, coneOffset2.Y, spot.pos.z+16));
        }
        
        // Change the marquee to the appropriate level
        setupMarqueeAndScreens(borderSector, mapInfo, mapCompleted);
        setupMarqueeAndScreens(backSector, mapInfo, mapCompleted, true);
        
        // Change floors for completion
        if (mapCompleted > 0)
        {
            skirtSector.SetTexture(0, TexMan.CheckForTexture("COMPBLUE", TexMan.Type_MiscPatch));
            spot.curSector.SetTexture(0, TexMan.CheckForTexture("RGATEBL1", TexMan.Type_MiscPatch));
            for (int i = 0; i < borderSector.lines.Size(); i++)
            { 
                if (TexMan.getName(borderSector.lines[i].sidedef[0].GetTexture(2)) == "RAMPBRDY") {
                    borderSector.lines[i].sidedef[0].SetTexture(2, TexMan.CheckForTexture("RAMPBRDB", TexMan.Type_MiscPatch));
                }
            }
        }

        // Find the attached screen sector and replace the screen with the level shot
        vector2 screenSectorVec = spot.Vec2Angle(60, spot.angle + 180);
        drawPortalMonitor(spot, screenSectorVec, "RSHOT" .. mapNumber, 0, 4.0);
        
        //Add to our dictionary so we can look up the number -> location later
        RampDataLibrary lib = RampDataLibrary.inst();
        lib.mapNumbersToPortals.Insert(mapNumber, spot);
        lib.mapNumbersToStationTags.Insert(mapNumber, spot.args[1]);
        return true;
    }
    
    static void setupMarqueeAndScreens(Sector borderSector, RampMapInfo mapInfo, int mapCompleted, int adjustScreens = false) {
        //DNLogger.Log("PORTALS", "Looking for marquee and screens bordering sector " .. borderSector.Index());
        
        for (int i = 0; i < borderSector.lines.Size(); i++)
        {
            let firstSidedef = borderSector.lines[i].sidedef[0];
            let secondSidedef = borderSector.lines[i].sidedef[1];
            if (firstSidedef && TexMan.getName(firstSidedef.GetTexture(2)) == "MARQX") {
                String marqueePrefix = "MARQ";
                if (mapCompleted > 0) {
                    marqueePrefix = "MARX";
                }
                firstSidedef.SetTexture(2, TexMan.CheckForTexture(marqueePrefix .. mapInfo.levelnum, TexMan.Type_MiscPatch));
            }
            if (adjustScreens) {
                //Length and difficulty - offset as appropriate
                if (firstSidedef && (TexMan.getName(firstSidedef.GetTexture(2)) == "RAMPRL")) {
                    firstSidedef.SetTextureXOffset(2, firstSidedef.GetTextureXOffset(2) + (mapInfo.length) * 32.0); 
                }
                if (firstSidedef && (TexMan.getName(firstSidedef.GetTexture(2)) == "RAMPRD")) {
                    firstSidedef.SetTextureXOffset(2, firstSidedef.GetTextureXOffset(2) + (mapInfo.difficulty) * 32.0);
                }
            }
            
            if (mapCompleted > 0)
            {
                if (firstSidedef && (TexMan.getName(firstSidedef.GetTexture(2)) == "RAMPBEMY")) {
                    firstSidedef.SetTexture(2, TexMan.CheckForTexture("RAMPBEMB", TexMan.Type_MiscPatch));
                }
                if (firstSidedef && (TexMan.getName(firstSidedef.GetTexture(2)) == "COMPYLM1")) {
                    firstSidedef.SetTexture(2, TexMan.CheckForTexture("COMPBLM1", TexMan.Type_MiscPatch));
                }
            }
        }        
    }
    
    static void drawPortalMonitor(RampPortal spot, Vector2 screenSectorVec, String textureName, int tileOffset = 0, double scale = 1.0) {
        textureId tx = TexMan.CheckForTexture(textureName, TexMan.Type_MiscPatch);
        if (tx.IsValid()) {
            let spawnedScreen = Actor.Spawn("MapSpot", (screenSectorVec.X, screenSectorVec.Y, spot.pos.z + 64));
            Sector screenSector = spawnedScreen.curSector;
            spawnedScreen.Destroy();
            for (int i = 0; i < screenSector.lines.Size(); i++)
            {
                if (TexMan.getName(screenSector.lines[i].sidedef[0].GetTexture(2)) == "RAMPSCRN") {
                        let sidedef = screenSector.lines[i].sidedef[0];
                        sidedef.SetTexture(2, tx);
                        sidedef.SetTextureXOffset(2, tileOffset * 64.0);
                        sidedef.SetTextureXScale(2, scale);
                        sidedef.SetTextureYScale(2, scale);
                        return;
                }
            }
            //Hmm, we didn't find a screen
            DNLogger.Log("ERROR", "Couldn't find a RAMPSCRN in sector " .. screenSector.Index() .. " for map " .. spot.tid .. ". Misplaced portal object?");
        }
    }

    static string setNavigationToMap(int mapnum)
    {
        //Remove old navigation markers
        RampNavigationMarker x; ThinkerIterator it = ThinkerIterator.Create("RampNavigationMarker");
        while (x = RampNavigationMarker(it.Next())) {
            x.Destroy();
        }

        //Retrieve portal from the map and create a new navigation marker there
        RampPortal spot = RampDataLibrary.inst().mapNumbersToPortals.Get(mapnum);
        if (!spot) {
            return "Hmm, can't find that one";
        }
        Actor.Spawn("RampNavigationMarker", (spot.pos.x, spot.pos.y, spot.pos.z + 64));

        //Now give the string to display
        string mapNumberString = "00" .. mapnum;
        mapNumberString = mapNumberString.Mid(mapNumberString.length() - 3, 3);
        string mapNumberKey = "" .. mapnum;
        if (mapNumberKey.length() == 1) {
            mapNumberKey = "0" .. mapNumberKey;
        }
        string output = "\cnMap #" .. mapNumberString .. "\n";
        
        string zoneString = "around here somewhere";
        int zoneKey = RampDataLibrary.Maps(mapnum).zone;
        if (zoneKey == 1) {zoneString = "a classic UAC base map"; }
        else if (zoneKey == 4) {zoneString = "a natural-themed map"; }
        else if (zoneKey == 5) {zoneString = "an ancient-themed map"; }
        else if (zoneKey == 6) {zoneString = "something a bit different"; }
        else if (zoneKey == 3) {zoneString = "going to take you to hell"; }
        else if (zoneKey == 2) {zoneString = "a castle map"; }
        else if (zoneKey == 7) {zoneString = "an urban map"; }
        else if (zoneKey == 0) {zoneString = "a futuristic map"; }

        int destinationStationTag = RampDataLibrary.inst().mapNumbersToStationTags.Get(mapnum);
        RampDataLibrary.SetRequestedStation(destinationStationTag);
        RampStationInfo stationInfo = RampDataLibrary.inst().stationTagsToInfo.GetIfExists(destinationStationTag);
        String stationString = "somewhere that doesn't have a name yet";
        if (stationInfo) {
            stationString = "at \cf" .. stationInfo.name;
        }
        
        output = output .. "\cf" .. Stringtable.Localize("$MAP" .. mapNumberKey .. "NAME") .. "\n\coby \cf" .. Stringtable.Localize("$MAP" .. mapNumberKey .. "AUTH") .. "\n\n\n\n";
        output = output .. "\coThis is " .. zoneString .. ",\n";
        if (destinationStationTag != RampDataLibrary.GetCurrentStation()) {
            output = output .. "\coand it's " .. stationString .. ".\n\n";
            output = output .. "\coLet's head there now - I've added a marker to your map too!";
        } else {
            output = output .. "\coand it's right here! I've marked it on your map.";
        }

        return output;
    }    
    
    static void UnlockNextStation() {
        RampDataLibrary.inst().stationsUnlocked = RampDataLibrary.inst().stationsUnlocked + 1;
        HubController.setupSubwayScreens();
    }
    
    static void UpdateMissionDisplay(bool clear = false) {

        DNLogger.Log("MISSION", "Updating mission display");
        RampMissionInfo info = RampDataLibrary.GetCurrentMissionInfo();
        if (info == null) {
            return;
        }
        Canvas cv = TexMan.GetCanvas("RAMPMTEX");
        if (!cv) return;
        cv.ClearScreen(Color(0, 0, 0));
        if (clear) {
            return;
        }

        if (RampDataLibrary.readInt("GameMode") == -1) {
            cv.DrawText(smallFont, Font.CR_LIGHTBLUE, 8, 8, "The game hasn't even\nstarted yet!");
            string gaugeText = "Get lost!";
            int x = 384 - 32 - bigFont.StringWidth(gaugeText);
            cv.DrawText(bigFont, Font.CR_LIGHTBLUE, x, 10, "Get lost!");
            return;
        }

        if (RampDataLibrary.readInt("GameMode") == 0) {
            string missionText = RampDataLibrary.inst().stationTagsToInfo.GetIfExists(info.stationTag).name;
            cv.DrawText(smallFont, Font.CR_ORANGE, 8, 8, missionText);
            string gaugeText = "Completed: " .. RampDataLibrary.getLevelsCompletedAtStation(info.stationTag).maps .. " / " .. RampDataLibrary.getLevelsCompletedAtStation(info.stationTag, true).maps;
            int x = 384 - 32 - bigFont.StringWidth(gaugeText);
            cv.DrawText(bigFont, Font.CR_ORANGE, x, 10, gaugeText);
            HubController.drawFreeModeDisplay();
            return;
        }
        
        let fontColor = Font.CR_GOLD;
        if (info.missionIsComplete()) {
            fontColor = Font.CR_LIGHTBLUE;
        }
        
        string missionText = RampDataLibrary.inst().stationTagsToInfo.GetIfExists(info.stationTag).name .. "\n" .. info.getMissionText();
        cv.DrawText(smallFont, fontColor, 8, 8, missionText);
        
        string gaugeText = info.getMissionCurrentValue() .. " / " .. info.getMissionTargetValue();
        int x = 384 - 32 - bigFont.StringWidth(gaugeText);
        cv.DrawText(bigFont, fontColor, x, 10, gaugeText);
    }
    
    static void DrawFreeModeDisplay() {
        
        let lib = RampDataLibrary.inst();
        
        Canvas cv = TexMan.GetCanvas("RAMPPROG");
        if (!cv) return;
        cv.ClearScreen(Color(0, 0, 0));
        
        //This is an array of the stations in alphabetical order
        Array<int> sortedStations;
        sortedStations.push(2011);
        sortedStations.push(2032);
        sortedStations.push(2026);
        sortedStations.push(2030);
        sortedStations.push(2009);
        sortedStations.push(2019);
        sortedStations.push(2010);
        sortedStations.push(2024);
        sortedStations.push(2038);
        sortedStations.push(2018);
        sortedStations.push(2034);
        sortedStations.push(2002);
        sortedStations.push(2029);
        sortedStations.push(2015);
        sortedStations.push(2004);
        sortedStations.push(2005);
        sortedStations.push(2020);
        sortedStations.push(2021);
        sortedStations.push(2017);
        sortedStations.push(2023);
        sortedStations.push(2003);
        sortedStations.push(2025);
        sortedStations.push(2037);
        sortedStations.push(2007);
        sortedStations.push(2008);
        sortedStations.push(2027);
        sortedStations.push(2022);
        sortedStations.push(2014);
        sortedStations.push(2039);
        sortedStations.push(2001);
        sortedStations.push(2031);
        sortedStations.push(2028);
        sortedStations.push(2013);
        sortedStations.push(2036);
        sortedStations.push(2035);
        sortedStations.push(2012);
        sortedStations.push(2006);
        sortedStations.push(2016);
        sortedStations.push(2033);
        
        int initialX = 30;
        int x = initialX;
        int initialY = 12;
        int y = initialY;
        
        int numberOfStations = lib.missionArray.size();
        for (int i = 0; i < numberOfStations; i++) {
            int stationTag = sortedStations[i];
            RampStationInfo stationInfo = RampDataLibrary.inst().stationTagsToInfo.GetIfExists(stationTag);
            int mapsCompleted = RampDataLibrary.getLevelsCompletedAtStation(stationTag).maps;
            int mapsAvailable = RampDataLibrary.getLevelsCompletedAtStation(stationTag, true).maps;
            
            RampMapInfo mapInfo = RampDataLibrary.GetFirstMapInfoAtStationTag(stationTag);
            if (mapInfo != null) {
                cv.DrawTexture(TexMan.CheckForTexture("RZONE" .. mapInfo.zone), false, x-10, y);
            }
            RampDataLibrary.GetLevelsAvailableAtStation(stationTag);
            cv.DrawText(smallFont, mapsCompleted == mapsAvailable ? Font.CR_LIGHTBLUE : Font.CR_GOLD, x, y, stationInfo.name);
            
            let pipX = x + 150;
            for (int j = 0; j < mapsAvailable; j++) {
                let c = Font.CR_LIGHTBLUE;
                if (j >= mapsCompleted) { c = Font.CR_GOLD; }
                cv.DrawText(smallFont, c, pipX, y, "I");
                pipX += 5;
            }
            
            y += 10;
            if (y > 140) {
                y = initialY;
                x += 225;
            }
        }
        
    }
    
    static void DrawEndingScreen() {
        
        let lib = RampDataLibrary.inst();
        
        Canvas cv = TexMan.GetCanvas("RAMPENDS");
        if (!cv) return;
        cv.ClearScreen(Color(0, 0, 0));
        
        int initialX = 18;
        int x = initialX;
        int initialY = 4;
        int y = initialY;
        
        int numberOfStations = lib.missionArray.size();
        int completedMapsDrawn = 0;
        for (int i = 1; i <= 311; i++) {
            let textColor = Font.CR_GOLD;
            RampMapInfo mapInfo = RampDataLibrary.Maps(i);
            int mapCompleted = RampDataLibrary.ReadInt("MapCompleted" .. i);
            
            textureid screenTexture = TexMan.CheckForTexture("RSHOT" .. i);
            if (!screenTexture.IsValid()) {
                screenTexture = TexMan.CheckForTexture("RAMPSCRN");
            }

            cv.DrawTexture(screenTexture, false, x, y, DTA_ScaleX, 0.125, DTA_ScaleY, 0.125, DTA_TopLeft, true, DTA_DestWidth, 512, DTA_DestHeight, 384);

            if (mapCompleted) {
                cv.DrawTexture(TexMan.CheckForTexture("RAMPDONE"), false, x, y);
                textColor = Font.CR_LIGHTBLUE;
                completedMapsDrawn++;
            }

            cv.DrawText(smallFont, textColor, x+1, y+1, i .. "");
            
            if (mapInfo != null) {
                cv.DrawTexture(TexMan.CheckForTexture("RZONE" .. mapInfo.zone), false, x+56, y+41);
            }

            x += 68;
            if (x >= 68 * 24) {
                x = initialX;
                y = y + 52;
            }
        }
        if (completedMapsDrawn == 311) {
            textureid screenTexture = TexMan.CheckForTexture("RALLCOMP");
            cv.DrawTexture(screenTexture, false, x, y, DTA_TopLeft, true);
        }
    }
    
   
    static string GetDestinationMessageForStation(int stationTag) {
        if (stationTag == 2040) {
            return "\cfCongratulations! Moving to the RAMP 2024 ending!";
        }
        RampStationInfo stationInfo = RampDataLibrary.inst().stationTagsToInfo.GetIfExists(stationTag);
        return "\coNow heading to \cf" .. stationInfo.name .. "\co!";
    }

    static string GetWelcomeMessageForCurrentStation() {
        RampDataLibrary lib = RampDataLibrary.inst();
        RampStationInfo stationInfo = lib.stationTagsToInfo.GetIfExists(RampDataLibrary.GetCurrentStation());
        RampMissionInfo missionInfo = lib.missionArray[RampDataLibrary.GetCurrentStationSlot()];
        
        if (RampDataLibrary.readInt("GameMode") == 0) {
            return "";
        }
        if (!stationInfo) {
            return "I have no idea what's happening";
        }
        
        string stationString = "\coWelcome to \cf" .. stationInfo.name .. "\co!\n\n";
        stationString = stationString .. "\coYour mission here is to\n";
        stationString = stationString .. "\co" .. missionInfo.getMissionText();
        
        return stationString;
    }
    
    static void UpdateStationLevers(int allOff = 0) {
        int stationSlot = RampDataLibrary.GetCurrentStationSlot();
        int leverOnTag = 2200 + stationSlot;
        if (allOff == 1) {
            DNLogger.log("TRAIN", "Setting no levers to ON");
        }
        for (int i = 2200; i < 2299; i++) {
            SectorTagIterator it = LevelLocals.CreateSectorTagIterator(i);
            int s = it.Next();
            if (s <= 0) { return; }
            let sec = Level.Sectors[s];
            sec.SetTexture(0, TexMan.CheckForTexture("RTRNBT1", TexMan.Type_MiscPatch));
            if (i == leverOnTag && allOff == 0) {
                DNLogger.log("TRAIN", "Setting lever " .. leverOnTag .. " to ON");
                sec.SetTexture(0, TexMan.CheckForTexture("RTRNBT2", TexMan.Type_MiscPatch));
            }
        }
    }
}

class LedController : Thinker {
    
    int x;
    int y;
    string text;
    int timer;
    int billboardTimer;
    
    static LedController inst(void)
	{
		ThinkerIterator it = ThinkerIterator.Create("LedController", STAT_DEFAULT);
		let p = LedController(it.Next());
		if (p) return p;
        return new("LedController").Init();
	}
    
    static clearscope LedController instOrNull(void)
    {
		ThinkerIterator it = ThinkerIterator.Create("LedController", STAT_DEFAULT);
		let p = LedController(it.Next());
		if (p) return p;
        return NULL;
    }
    
    void setText(string text) {
        self.text = text;
        self.timer = 0;
        self.x = 0;
        self.y = 0;
    }
    
    static void setStationDisplay(bool next, int stationTag) {
        RampStationInfo stationInfo = RampDataLibrary.inst().stationTagsToInfo.GetIfExists(stationTag);
        if (stationInfo) {
            string textToDisplay = stationInfo.name;
            if (next) {
                textToDisplay = "Next: " .. textToDisplay;
            }
            LedController.inst().setText(textToDisplay);
        }
    }
    
    LedController Init() {
        int stationTag = RampDataLibrary.GetCurrentStation();
        LedController.setStationDisplay(false, stationTag);
        return self;
    }
    
    override void Tick() {
        handleTrainMarquee();
        handleBackScreen();
    }
    
    void handleTrainMarquee() {
        Canvas cv = TexMan.GetCanvas("RAMPMARQ");
        if (!cv) return;
        cv.ClearScreen(Color(0, 0, 0));
        
        if (!self.text) {
            return;
        }
        
        if (timer < 36) {
            y = 10 - (timer/4);
        }
        else {
            y = 1;
            if (timer > 50) {
                x -= 3;
                if (x < -200) {
                    x = 144;
                }
            }
        }

        cv.DrawText(smallFont, Font.CR_WHITE, x, y, text);
        self.timer++;        
    }
    
    void handleBackScreen() {
        self.billboardTimer--;
        if (self.billboardTimer > 5) {
            return;
        }
        Canvas cv = TexMan.GetCanvas("RAMPBILL");
        if (!cv) return;
        cv.ClearScreen(Color(0,0,0));
        if (self.billboardTimer > 0) {
            return;
        }
        
        //If everything's completed, just draw the trophy screen
        if (RampDataLibrary.inst().mapsCompleted == 311) {
            textureid screenTexture = TexMan.CheckForTexture("RALLCOM2");
            cv.DrawTexture(screenTexture, false, x, y, DTA_TopLeft, true);
            return;
        }
        int mapToDisplay = random(1, 311);
        textureid screenTexture;
        screenTexture.SetInvalid();
        int tries = 0;
        while (!screenTexture.IsValid() && tries < 500) {
            textureid screenTexture = TexMan.CheckForTexture("RSHOT" .. mapToDisplay);
            if (screenTexture.IsValid()) {
                cv.DrawTexture(screenTexture, false, 0, 0, DTA_TopLeft, true);
                
                let fontColor = Font.CR_GOLD;
                if (RampDataLibrary.ReadInt("MapCompleted" .. mapToDisplay)) {
                    fontColor = Font.CR_LIGHTBLUE;
                }
                
                self.DrawTextOnBackScreen(cv, bigfont, fontColor, 336, "Map " .. mapToDisplay);
                self.DrawTextOnBackScreen(cv, bigfont, fontColor, 352, Stringtable.Localize("$MAP" .. mapToDisplay .. "NAME"));
                self.DrawTextOnBackScreen(cv, bigfont, fontColor, 368, Stringtable.Localize("$MAP" .. mapToDisplay .. "AUTH"));
                
                self.billboardTimer = 350;
                return;
            }
            mapToDisplay++;
            tries++;
            if (mapToDisplay > 311) { mapToDisplay = 1; }
        }
    }
    
    void drawTextOnBackScreen(Canvas cv, Font font, Color color, int y, string text) {
        cv.DrawText(font, color, 256 - font.StringWidth(text)/2, y, text);
    }
}