class RampDataLibrary : Thinker
{
    //Level status index corresponds to map number
    int[400] levelSecretsFound;
    int[400] causewaySlotsToMapNums;
    
    int existentMaps;
    
    int SECRET_VALUE;
    int LEVEL_VALUE_PERCENT;
    int MAX_STATION_TAG;
    int MAX_VIEW_TAG;
    int MAX_STATION_SLOTS;
    
    int mapsCompleted;
    int stationsUnlocked;
    string typedMapNumber;
    
    Dictionary dic;
    Map<int, RampPortal> mapNumbersToPortals;
    Array<RampMapInfoList> zonesToMapInfoLists;
    Map<int, RampMapInfo> mapNumbersToInfo;
    Map<int, int> mapNumbersToStationTags;
    Map<int, RampStationInfo> stationTagsToInfo;
    Array<RampMissionInfo> missionArray;
    
    static clearscope int ReadInt(String key) { return RampDataLibrary.instOrNull().dic.At(key).ToInt(); }
    static clearscope string ReadString(String key) { return RampDataLibrary.instOrNull().dic.At(key); }
    static clearscope void WriteString(String key, String value) { RampDataLibrary.instOrNull().dic.Insert(key, value); }
    static clearscope void WriteInt(String key, int value) { RampDataLibrary.instOrNull().dic.Insert(key, value .. ""); }
    
    static clearscope RampMapInfo Maps(int mapNumber) {
        return RampDataLibrary.instOrNull().mapNumbersToInfo.GetIfExists(mapNumber);
    }
    
    static RampMapInfo GetNextMapInfoForZone(int zone) {
        return RampDataLibrary.inst().zonesToMapInfoLists[zone].getNext();
    }

    //We want to be able to get the RampDataLibrary instance from both play and UI contexts.
    //Thinker is play-scoped, so we can't create a new instance if we call this from a UI
    //context - the conversation menu has to use GetInstance instead
    
    static RampDataLibrary inst(void)
	{
		ThinkerIterator it = ThinkerIterator.Create("RampDataLibrary", STAT_STATIC);
		let p = RampDataLibrary(it.Next());
		if (p) return p;
        DNLogger.Log("INIT", "Initializing RAMP data library");
        return new("RampDataLibrary").Init();
	}
    
    static clearscope RampDataLibrary instOrNull(void)
    {
		ThinkerIterator it = ThinkerIterator.Create("RampDataLibrary", STAT_STATIC);
		let p = RampDataLibrary(it.Next());
		if (p) return p;
        return NULL;
    }
    
    //Quick, make this a static thinker when we initialize
    RampDataLibrary Init(void)
	{
		ChangeStatNum(STAT_STATIC);
        dic = Dictionary.Create();
        dic.Insert("currentMap", "0");
        dic.Insert("cashOnHand", "0");
        dic.Insert("totalCashEarned", "0");
        dic.Insert("hubValue", "0");
        dic.Insert("GameMode", "-1");
        
        SECRET_VALUE = 250;
        LEVEL_VALUE_PERCENT = 100;
        MAX_STATION_TAG = 2039;
        MAX_VIEW_TAG = 2111;
        MAX_STATION_SLOTS = MAX_STATION_TAG - 2000;
        stationsUnlocked = 1;

		return self;
	}
    
    //Get the RAMPDATA lump prepared by the site and interpret it into data for later use
    void refreshDataArrays()
    {
        DNLogger.Log("INIT", "Refreshing data arrays");
        //If our zones to portal lists map is currently empty, this is the first run.
        //We need to know this so that we decide whether to shuffle them
        if (self.zonesToMapInfoLists.Size() == 0) {
            DNLogger.Log("INIT", "No zone to map info list map, setting it up");
            for (int i = 0; i < 8; i++) {
                self.zonesToMapInfoLists.Push(new("RampMapInfoList"));
            }
        }
        
        //Reset our count of maps and our maps per zone
        existentMaps = 0;
        DictionaryIterator it = DictionaryIterator.Create(dic);
        while (it.Next()) {
            if (it.Key().Left(11) == "MapsPerZone") {
                dic.Insert(it.Key(), "0");
            }
        }
        
        Map<int, RampMapInfoList> zonesToNewMapLists;
        for (int i = 0; i < 8; i++) {
            zonesToNewMapLists.Insert(i, new("RampMapInfoList"));
        }
        
        int debug_limit = 99999;
        //Get the map data from the data file and add it to our registry...
        Array<String> lines; RampDataLibrary.getLumpData("RAMPDATA").Split(lines, "\n");
        for (int i = 0; i < lines.Size(); i++)
        {
            String line = lines[i];
            if (line.Length() < 2) { continue; }
            Array<String> lineData; line.Split(lineData, ",");
            //Arrays are 1-based to line up with map slots!
            let info = RampMapInfo.Create(
                lineData[7],         // Zone
                lineData[4].ToInt(), // Length
                lineData[5].ToInt(), // Difficulty
                lineData[6].ToInt(), // Monsters
                lineData[2].ToInt(), // Can jump
                lineData[3].ToInt(), // Is WIP
                lineData[0].ToInt(), // levelnum
                lineData[1]          // Lump name
            );
            
            //Count this map in the global and zone lists
            int mapnumber = lineData[0].ToInt();
            self.mapNumbersToInfo.Insert(mapnumber, info);
            existentMaps++;
            int zoneCount = dic.At("MapsPerZoneTotal" .. info.zone).ToInt();
            dic.Insert("MapsPerZoneTotal" .. info.zone, (zoneCount+1) .. "");
            
            // If this map doesn't already have a portal, put it in the map of maps to assign.
            if (!RampDataLibrary.readInt("MapAssignedToPortal" .. mapnumber)) {
                DNLogger.Log("PORTALS", "Adding map " .. mapnumber .. " to the new maps lists as it has no portal assigned");
                zonesToNewMapLists.Get(info.zone).push(info);
                debug_limit--;
            }
            if (debug_limit <= 0) {
                break;
            }
        }
        
        // Shuffle our new lists, and if we have any, add the new mapinfos to the master list.
        for (int i = 0; i < 8; i++) {
            RampMapInfoList mapInfosThisZone = zonesToNewMapLists.GetIfExists(i);   
            mapInfosThisZone.shuffle();
            mapInfosThisZone.sortByCompleteness();
            for (int x = 0; x < mapInfosThisZone.size(); x++) {
                let mapInfo = mapInfosThisZone.get(x);
                DNLogger.Log("PORTALS", "Adding map to the map info lists: " .. mapInfo.toString());
                RampDataLibrary.writeInt("MapAssignedToPortal" .. mapInfo.levelnum, 1);
                self.zonesToMapInfoLists[i].push(mapInfo);
            }
            self.zonesToMapInfoLists[i].rewind();
        }
        DNLogger.Log("INIT", "Existent maps: " .. existentMaps);
        
        // Populate our station info
        Array<String> stationLines; RampDataLibrary.getLumpData("RAMPSTTN").Split(stationLines, "\n");
        for (int i = 0; i < stationLines.Size(); i++) {
            String line = stationLines[i];
            if (line.Length() < 2) { continue; }
            Array<String> lineData; line.Split(lineData, ",");
            let info = RampStationInfo.Create(lineData[1], lineData[2]);
            info.name.StripLeftRight();
            info.sky.StripLeftRight();
            int stationTag = lineData[0].ToInt();
            self.stationTagsToInfo.insert(stationTag, info);
        }
    }
    
    void updateStationCounts() {
        //Clear our points first
        for (int i = 2001; i <= MAX_STATION_TAG; i++) {
            stationTagsToInfo.GetIfExists(i).clearPoints();
        }
        //Repopulate from the latest portal to station tags
        MapIterator<int, int> it;
        if (!it.Init(mapNumbersToStationTags)) {
            DNLogger.log("INIT", "Station count iterator was invalid");
            return;
        }
        while (it.Next()) {
            int levelnum = it.GetKey(); int stationTag = it.GetValue();
            RampMapInfo mapInfo = mapNumbersToInfo.GetIfExists(levelnum);
            if (mapInfo == null) {
                DNLogger.log("INIT", "Map data object was invalid: " .. levelnum);
                return;
            }
            stationTagsToInfo.GetIfExists(stationTag).lengthPointsAvailable += mapInfo.length;
            stationTagsToInfo.GetIfExists(stationTag).difficultyPointsAvailable += mapInfo.difficulty;
            stationTagsToInfo.GetIfExists(stationTag).pointsAvailable += (mapInfo.difficulty + mapInfo.length);
            stationTagsToInfo.GetIfExists(stationTag).monstersAvailable += mapInfo.monsters;
        }
        DNLogger.log("INIT", "Filled station info stats");
        MapIterator<int, RampStationInfo> it2;
        it2.Init(stationTagsToInfo);
        while (it2.Next()) {
            RampStationInfo info = it2.GetValue();
            DNLogger.log("INIT", info.toString());
        }
    }
    
    Array<int> stationTagsForPickingMissions;
    void createMissionArray() {
        //Starting with a requirement for 8 non-WIP maps, create a list of station tags and then pick missions from it
        int nonWipLevelRequirement = 10;
        while (nonWipLevelRequirement >= 0) {
            for (int i = 2001; i <= MAX_STATION_TAG; i++) {
                int nonWipMapsAvailable = GetNonWipLevelsAvailableAtStation(i).maps;
                if (nonWipMapsAvailable == nonWipLevelRequirement) {
                    DNLogger.log("MISSION", "Non-WIP levels at " .. i .. ": " .. nonWipMapsAvailable);
                    stationTagsForPickingMissions.push(i);
                }
            }
            while (stationTagsForPickingMissions.size() > 0) {
                pickMission();
            }
            nonWipLevelRequirement--;
        }
        DNLogger.Log("INIT", "Mission array has " .. missionArray.size() .. " stations");
        return;
    }
    
    void pickMission() {
        int index = Random(0, stationTagsForPickingMissions.Size()-1);
        int stationTag = stationTagsForPickingMissions[index];
        stationTagsForPickingMissions.Delete(index);
        
        //Set a mission for this map. Make it depend on the length through the game to some extent
        int missionType = Random(1, 5);
        int missionTarget = 0;
        if (missionType == 1) { // Complete a certain number of maps
            if (missionArray.size() < 6) { missionTarget = Random(2, 4); }
            else if (missionArray.size() < 12) { missionTarget = Random(2, 5); }
            else if (missionArray.size() < 18) { missionTarget = Random(3, 5); }
            else if (missionArray.size() < 24) { missionTarget = Random(3, 6); }
            else if (missionArray.size() < 30) { missionTarget = Random(4, 7); }
            else { missionTarget = Random(5, 7); }
        }
        if (missionType == 2 || missionType == 3 || missionType == 4) { // Get X points
            if (missionArray.size() < 6) { missionTarget = Random(20, 50); }
            else if (missionArray.size() < 18) { missionTarget = Random(25, 60); }
            else if (missionArray.size() < 30) { missionTarget = Random(40, 80); }
            else { missionTarget = Random(60, 85); }
        }
        if (missionType == 5) { // Kill X monsters
            if (missionArray.size() < 6) { missionTarget = Random(10, 30); }
            else if (missionArray.size() < 18) { missionTarget = Random(25, 50); }
            else if (missionArray.size() < 30) { missionTarget = Random(40, 70); }
            else { missionTarget = Random(60, 90); }
        }
        missionArray.push(RampMissionInfo.create(stationTag, Random(0, 3) * 80, missionType, missionTarget));
        DNLogger.Log("INIT", "Pushing: " .. stationTag);
    }
    
    static bool currentStationMissionIsComplete() {
        if (RampDataLibrary.readInt("GameMode") <= 0) {
            return false; //No missions under game mode 0
        }
        RampMissionInfo info = RampDataLibrary.inst().missionArray[RampDataLibrary.GetCurrentStationSlot()];
        DNLogger.log("MISSION", "Checking station at slot " .. (RampDataLibrary.inst().stationsUnlocked - 1) .. ": " .. info.stationTag);
        return info.missionIsComplete();
    }
    
    static bool latestStationMissionIsComplete() {
        if (RampDataLibrary.readInt("GameMode") <= 0) {
            return false; //No missions under game mode 0
        }
        int stationsUnlocked = RampDataLibrary.inst().stationsUnlocked;
        RampMissionInfo info = RampDataLibrary.inst().missionArray[stationsUnlocked - 1];
        DNLogger.log("MISSION", "Checking station at slot " .. (stationsUnlocked - 1) .. ": " .. info.stationTag);
        return info.missionIsComplete();
    }
    
    static int getStationsUnlocked() {
        return RampDataLibrary.inst().stationsUnlocked;
    }

    static String getLumpData(String lumpname)
    {
        int lumpindex = Wads.FindLump(lumpname, 0, 0);
        String lumpdata = Wads.ReadLump(lumpindex);
        return lumpdata;
    }

    static int GetCurrentMap(Actor activator)                { return ReadInt("currentMap"); }
    static void SetCurrentMap(Actor activator, int levelNum) { WriteInt("currentMap", levelNum); }
    
    static int GetCurrentStation()                  { return ReadInt("currentStation"); }
    static void SetCurrentStation(int stationTag) {
        RampDataLibrary.inst().typedMapNumber = "";
        DNLogger.log("TRAIN", "Updated current station tag to " .. stationTag);
        WriteInt("currentStation", stationTag);
        HubController.UpdateMissionDisplay(true);
        //Find the slot number for this station as well and update that
        for (int i = 0; i < RampDataLibrary.inst().missionArray.Size(); i++) {
            RampMissionInfo info = RampDataLibrary.inst().missionArray[i];
            if (info.stationTag == stationTag) {
                WriteInt("currentStationSlot", i);
                DNLogger.log("TRAIN", "Updated current station slot to " .. i);
                HubController.UpdateMissionDisplay();
                HubController.UpdateStationLevers();
                return;
            }
        }
        DNLogger.log("ERROR", "No station slot found...");
    }

    static int GetCurrentStationSlot()              { return ReadInt("currentStationSlot"); }
    static RampMissionInfo GetCurrentMissionInfo()  { return RampDataLibrary.instOrNull().missionArray[ReadInt("currentStationSlot")]; }
    
    static int GetRequestedStation()                { return ReadInt("requestedStation"); }
    static void SetRequestedStation(int stationTag) { WriteInt("requestedStation", stationTag); }
    
    static int GetNotRequestedStation() {
        int requestedStation = ReadInt("requestedStation");
        int currentStation = ReadInt("currentStation");
        int returnStation = requestedStation;
        while (returnStation == requestedStation || returnStation == currentStation) {
            returnStation = random(2001, RampDataLibrary.inst().MAX_STATION_TAG);
        }
        return returnStation;
    }
    
    static int GetAnyOpenStation() {
        
        int missionArraySize = RampDataLibrary.instOrNull().missionArray.size();
        int upperBound = missionArraySize;
        int stationsUnlocked = RampDataLibrary.instOrNull().stationsUnlocked;
        if (upperBound > stationsUnlocked) {
            upperBound = stationsUnlocked;
        }
        
        int index = Random(0, upperBound-1);
        return RampDataLibrary.instOrNull().missionArray[index].stationTag;
    }
    
    static int GetRandomView() {
        return random(2101, RampDataLibrary.inst().MAX_VIEW_TAG);
    }
    
    static clearscope RampCompletionInfo GetLevelsCompletedAtStation(int stationTag, bool justAllMaps = false) {
        return GetLevels(stationTag, justAllMaps);
    }
    
    static clearscope RampCompletionInfo GetLevelsAvailableAtStation(int stationTag) {
        return GetLevels(stationTag, true);
    }
    
    static clearscope RampCompletionInfo GetNonWipLevelsAvailableAtStation(int stationTag) {
        return GetLevels(stationTag, true, true);
    }
    
    static clearscope RampCompletionInfo GetLevels(int stationTag = 0, bool justAllMaps = false, bool notWip = false)
    {
        RampCompletionInfo completionInfo = new("RampCompletionInfo");
        MapIterator<int, RampMapInfo> it;
        if (!it.Init(RampDataLibrary.instOrNull().mapNumbersToInfo)) {
            DNLogger.log("GETLEVELS", "Iterator was invalid");
            return completionInfo;
        }
        while (it.Next()) {
            int key = it.GetKey(); RampMapInfo info = it.GetValue();
            if (notWip && info.isWip) {
                continue;
            }
            int infoStationTag = RampDataLibrary.instOrNull().mapNumbersToStationTags.GetIfExists(info.levelnum);
            if (stationTag == 0 || infoStationTag == stationTag) {
                DNLogger.log("GETLEVELS", "Found map " .. info.levelnum .. " at station " .. stationTag);
                bool isCompleted = (RampDataLibrary.ReadInt("MapCompleted" .. info.levelnum) > 0);
                completionInfo.maps +=             (justAllMaps || isCompleted) ? 1 : 0;
                completionInfo.points +=           (justAllMaps || isCompleted) ? (info.difficulty + info.length) : 0;
                completionInfo.difficultyPoints += (justAllMaps || isCompleted) ? info.difficulty : 0;
                completionInfo.lengthPoints +=     (justAllMaps || isCompleted) ? info.length : 0;
                completionInfo.monsters +=         (justAllMaps || isCompleted) ? RampDataLibrary.ReadInt("MapTotalKilledMonsters" .. info.levelnum) : 0;
            }
        }
        DNLogger.log("GETLEVELS", completionInfo.maps .. " maps" .. (justAllMaps ? " " : " completed ") .. "at station " .. stationTag);
        return completionInfo;
    }
    
    static clearscope RampMapInfo GetFirstMapInfoAtStationTag(int stationTag)
    {
        MapIterator<int, RampMapInfo> it;
        if (!it.Init(RampDataLibrary.instOrNull().mapNumbersToInfo)) {
            DNLogger.log("GETLEVELS", "Iterator was invalid");
            return null;
        }
        while (it.Next()) {
            int key = it.GetKey(); RampMapInfo info = it.GetValue();
            int infoStationTag = RampDataLibrary.instOrNull().mapNumbersToStationTags.GetIfExists(info.levelnum);
            if (infoStationTag == stationTag) {
                return info;
            }
        }
        return null;
    }
    
    static int GetStationForMap(int mapNumber) {
        return RampDataLibrary.instOrNull().mapNumbersToStationTags.GetIfExists(mapNumber);
    }
    
    static string GetSkyForStation(int stationTag) {
        return RampDataLibrary.instOrNull().stationTagsToInfo.GetIfExists(stationTag).sky;
    }
    
    static int GetStationInSlotIfAllowed(int slotNumber) {
        if (slotNumber >= RampDataLibrary.instOrNull().stationsUnlocked && RampDataLibrary.readInt("GameMode") != 0) {
            return -1;
        }
        return RampDataLibrary.instOrNull().missionArray[slotNumber].stationTag;
    }
    
    static clearscope int GetSlotIfStationOpen(int stationTag) {
        DNLogger.Log("TRAIN", "Looking for " .. stationTag .. " being open");
        int stationsUnlocked = RampDataLibrary.instOrNull().stationsUnlocked;
        if (RampDataLibrary.readInt("GameMode") == 0) {
            DNLogger.Log("TRAIN", "In free roaming mode, so all stations will be open");
            stationsUnlocked = 999;
        }
        for (int i = 0; i < stationsUnlocked && i < RampDataLibrary.instOrNull().missionArray.size(); i++) {
            if (RampDataLibrary.instOrNull().missionArray[i].stationTag == stationTag) {
                return i;
            }
        }
        DNLogger.Log("TRAIN", "Station tag " .. stationTag .. " was not open");
        return -1;
    }

    //Bit of help on classes!
    static bool isAKindOf(string className, string classNameToMatch) {
        if (className == classNameToMatch) {
            return true;
        }
        class<Actor> cls = className;
        if (cls == null) {
            return false;
        }
        class<Actor> rootclass = Actor.GetReplacee(cls);
        if (rootclass.getClassName() == classNameToMatch) {
            return true;
        }
        return false;
    }
   
    static void outputDictionary()
    {
        Dictionary theDictionary = RampDataLibrary.inst().dic;
        DictionaryIterator d = DictionaryIterator.Create(theDictionary);
        console.printf("\caDictionary Contents");
        while (d.Next()) {    
            console.printf("\ck%-30s \cl%s", d.Key(), d.Value());
        }
    }

    static int tracePressedMapButton(Actor activator) {
        FLineTraceData Results;
        bool hit = activator.LineTrace(activator.angle, 64, activator.pitch, flags: TRF_ALLACTORS, offsetz: activator.height-16, data: Results);        
        if (!hit || Results.HitType != TRACE_HitActor) {
            return 0;
        }
        
        int buttonNumber = Results.HitActor.args[2];
        DNLogger.Log("KEYPAD", "Hit button " .. buttonNumber);
        if (buttonNumber == 11) { buttonNumber = 0; }

        activator.A_StartSound("Phone/Touchtone" .. buttonNumber);
        
        if (buttonNumber == 10) {
            RampDataLibrary.inst().typedMapNumber = "";
            return 0;
        }
        string currentTypedNumber = RampDataLibrary.inst().typedMapNumber;
        if (buttonNumber != 12) {
            currentTypedNumber = currentTypedNumber .. buttonNumber;
        }
        
        if (currentTypedNumber.ToInt() >= 32 || currentTypedNumber.length() >= 3 || buttonNumber == 12) {
            DNLogger.Log("KEYPAD", "Button string accepted: " .. currentTypedNumber);
            RampDataLibrary.inst().typedMapNumber = "";
            int number = currentTypedNumber.ToInt(10);
            DNLogger.Log("KEYPAD", "Button string translated: " .. number);
            return number;
        }
        RampDataLibrary.inst().typedMapNumber = currentTypedNumber;
        DNLogger.Log("KEYPAD", "Button string is " .. currentTypedNumber);
        return 0;
    }
    
    static void WriteFinalStats(Actor activator) {
        RampDataLibrary dataLibrary = RampDataLibrary.inst();
        //Only update the game completion time if we're here for the first time
        if (RampDataLibrary.ReadInt("GameCompletionTime") == 0) {
            RampDataLibrary.WriteInt("GameCompletionTime", Level.TotalTime);
            DNLogger.Log("END", "Game completion time: " .. RampDataLibrary.ReadInt("GameCompletionTime"));
        }
        UpdateGameCompletionBoard(RampDataLibrary.ReadInt("GameCompletionTime"));
        RampDataLibrary.WriteInt("MapsCompleted", RampDataLibrary.inst().mapsCompleted);
        DNLogger.Log("END", "Maps completed: " .. RampDataLibrary.ReadInt("MapsCompleted"));
        UpdateLevelCompletionBoard(RampDataLibrary.ReadInt("MapsCompleted"));
        
        int slotnum = 1;
        int i; SectorTagIterator it = LevelLocals.CreateSectorTagIterator(992);
        while ((i = it.Next()) && i != -1) {
            Sector s = Level.Sectors[i];
            DNLogger.Log("END", "Got sector " .. s.Index());
            int mapnum = dataLibrary.causewaySlotsToMapNums[slotnum];
            

            Line backWall = GetLineByMidTexture(s, "RSHOT1");
            if (backWall == null) {
                continue; //Probably already done this, so skip it
            }
            backWall.sidedef[0].SetTexture(1, TexMan.CheckForTexture("RSHOT" .. mapnum, TexMan.Type_MiscPatch));
            
            Line frontWall = GetTwoSidedLine(s);
            if (frontWall == null || slotnum > dataLibrary.existentMaps) {
                DNLogger.Log("ERROR", "Error placing picture for slot " .. slotnum .. " in sector " .. i);
                continue;
            }

            frontWall.sidedef[0].SetTexture(1, TexMan.CheckForTexture("RFRAME"));
            frontWall.sidedef[0].SetTextureXOffset(1, 0.0);
            frontWall.sidedef[0].SetTextureYOffset(1, 0.0);
            frontWall.sidedef[1].SetTexture(1, TexMan.CheckForTexture("RFRAME"));
            frontWall.sidedef[1].SetTextureXOffset(1, 0.0);
            frontWall.sidedef[1].SetTextureYOffset(1, 0.0);
            if (RampDataLibrary.ReadInt("MapCompleted" .. mapnum) == 1) {
                frontWall.sidedef[0].SetTexture(1, TexMan.CheckForTexture("RFRAMEDN"));
                frontWall.sidedef[1].SetTexture(1, TexMan.CheckForTexture("RFRAMEDN"));
            }
            slotnum++;
        }
        RampDataLibrary.WriteInt("PlacedGalleryPictures", 1); 
    }
    
    static void UpdateGameCompletionBoard(int ticsTaken) {
       int hoursTaken = 0;
       int minutesTaken = 0;
       int secondsTaken = ticsTaken / 35;
       while (secondsTaken > 3600) { hoursTaken += 1; secondsTaken -= 3600; }
       while (secondsTaken > 60) { minutesTaken += 1; secondsTaken -= 60; }
       DNLogger.Log("END", "Hours taken: " .. hoursTaken);
       DNLogger.Log("END", "Minutes taken: " .. minutesTaken);
       DNLogger.Log("END", "Seconds taken: " .. secondsTaken);
        
       Line l = GetLineWithTag(504);
       l.sidedef[0].SetTextureXOffset(1, 44 * (hoursTaken/100));
       l = GetLineWithTag(505);
       l.sidedef[0].SetTextureXOffset(1, 44 * ((hoursTaken/10) % 10));
       l = GetLineWithTag(506);
       l.sidedef[0].SetTextureXOffset(1, 44 * (hoursTaken % 100));
       
       l = GetLineWithTag(507);
       l.sidedef[0].SetTextureXOffset(1, 44 * (minutesTaken/10));
       l = GetLineWithTag(508);
       l.sidedef[0].SetTextureXOffset(1, 44 * (minutesTaken % 10));

       l = GetLineWithTag(509);
       l.sidedef[0].SetTextureXOffset(1, 44 * (secondsTaken/10));
       l = GetLineWithTag(510);
       l.sidedef[0].SetTextureXOffset(1, 44 * (secondsTaken % 10));
    }
    
    static void UpdateLevelCompletionBoard(int levels) {
       Line l = GetLineWithTag(501);
       l.sidedef[0].SetTextureXOffset(1, 44 * (levels/100));
       l = GetLineWithTag(502);
       l.sidedef[0].SetTextureXOffset(1, 44 * ((levels/10) % 10));
       l = GetLineWithTag(503);
       l.sidedef[0].SetTextureXOffset(1, 44 * (levels % 100));
    }
    
    static Line GetLineWithTag(int tag) {
        int l; LineIdIterator it = LevelLocals.CreateLineIdIterator(tag); l = it.Next();
        if (l == -1) return null;
        return Level.Lines[l];
    }
    
    static Line GetLineByMidTexture(Sector s, string textureToDetect) {
        for (int i = 0; i < s.lines.Size(); i++)
        {
            if (TexMan.getName(s.lines[i].sidedef[0].GetTexture(1)) == textureToDetect) {
                return s.lines[i];
            }
        }
        return null;
    }
    
    static Line GetTwoSidedLine(Sector s) {
        for (int i = 0; i < s.lines.Size(); i++)
        {
            if (s.lines[i].sidedef[1] != null) {
                return s.lines[i];
            }
        }
        return null;
    }
   
    static void RecordArrivalAtCurrentStation() {
        int stationTag = RampDataLibrary.GetCurrentStation();
        RampDataLibrary.writeInt("ArrivedAtStation" .. stationTag, 1);
    }
    
    static int HasVisitedCurrentStation() {
        return RampDataLibrary.readInt("ArrivedAtStation" .. RampDataLibrary.GetCurrentStation());
    }
}
